<?php
/**
 * Infinity Medical - Installation Verification Script
 * Use this to verify your installation is working correctly
 */

error_reporting(E_ALL);
ini_set('display_errors', 1);

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Installation Verification - Infinity Medical</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body { font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif; background: #f5f5f5; padding: 20px; }
        .container { max-width: 900px; margin: 0 auto; background: white; border-radius: 10px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); padding: 30px; }
        h1 { color: #667eea; margin-bottom: 10px; }
        h2 { color: #333; margin: 20px 0 10px 0; border-bottom: 2px solid #667eea; padding-bottom: 5px; }
        .check { padding: 10px; margin: 5px 0; border-radius: 5px; }
        .success { background: #e8f5e9; color: #2e7d32; border-left: 4px solid #4CAF50; }
        .error { background: #ffebee; color: #c62828; border-left: 4px solid #f44336; }
        .warning { background: #fff3e0; color: #e65100; border-left: 4px solid #ff9800; }
        .info { background: #e3f2fd; color: #1565c0; border-left: 4px solid #2196f3; }
        code { background: #f5f5f5; padding: 2px 6px; border-radius: 3px; font-family: 'Courier New', monospace; }
        .section { margin: 20px 0; }
        table { width: 100%; border-collapse: collapse; margin: 10px 0; }
        table td { padding: 8px; border-bottom: 1px solid #e0e0e0; }
        table td:first-child { font-weight: bold; width: 200px; }
        .btn { display: inline-block; padding: 10px 20px; background: #667eea; color: white; text-decoration: none; border-radius: 5px; margin: 10px 5px 0 0; }
        .btn:hover { background: #5568d3; }
    </style>
</head>
<body>
    <div class="container">
        <h1>🔍 Infinity Medical - Installation Verification</h1>
        <p style="color: #666; margin-bottom: 30px;">This script verifies that your installation is configured correctly.</p>

        <?php
        $allChecks = [];
        
        // 1. Check PHP Version
        echo '<div class="section">';
        echo '<h2>1. PHP Version</h2>';
        $phpVersion = phpversion();
        $phpOk = version_compare($phpVersion, '7.4.0', '>=');
        echo '<div class="check ' . ($phpOk ? 'success' : 'error') . '">';
        echo ($phpOk ? '✅' : '❌') . ' PHP Version: ' . $phpVersion;
        if (!$phpOk) echo ' (Required: PHP 7.4+)';
        echo '</div>';
        $allChecks[] = $phpOk;
        echo '</div>';

        // 2. Check Required Extensions
        echo '<div class="section">';
        echo '<h2>2. PHP Extensions</h2>';
        $extensions = ['pdo', 'pdo_mysql', 'mbstring', 'json', 'curl'];
        $extOk = true;
        foreach ($extensions as $ext) {
            $loaded = extension_loaded($ext);
            $extOk = $extOk && $loaded;
            echo '<div class="check ' . ($loaded ? 'success' : 'error') . '">';
            echo ($loaded ? '✅' : '❌') . ' ' . $ext;
            echo '</div>';
        }
        $allChecks[] = $extOk;
        echo '</div>';

        // 3. Check Configuration File
        echo '<div class="section">';
        echo '<h2>3. Configuration Files</h2>';
        $configExists = file_exists(__DIR__ . '/api/config/env.php');
        echo '<div class="check ' . ($configExists ? 'success' : 'error') . '">';
        echo ($configExists ? '✅' : '❌') . ' Configuration file (api/config/env.php)';
        echo '</div>';
        
        if ($configExists) {
            require_once __DIR__ . '/api/config/env.php';
            $configOk = defined('DB_HOST') && defined('DB_NAME') && defined('DB_USER');
            echo '<div class="check ' . ($configOk ? 'success' : 'warning') . '">';
            echo ($configOk ? '✅' : '⚠️') . ' Configuration constants loaded';
            echo '</div>';
            $allChecks[] = $configOk;
        } else {
            $allChecks[] = false;
        }
        echo '</div>';

        // 4. Database Connection
        echo '<div class="section">';
        echo '<h2>4. Database Connection</h2>';
        if ($configExists && defined('DB_HOST')) {
            try {
                $dsn = "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=utf8mb4";
                $pdo = new PDO($dsn, DB_USER, DB_PASS);
                $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
                
                echo '<div class="check success">✅ Database connection successful</div>';
                
                // Check tables
                $tables = ['services', 'ser', 'rental', 'parts'];
                $tablesOk = true;
                foreach ($tables as $table) {
                    $stmt = $pdo->query("SHOW TABLES LIKE '$table'");
                    $exists = $stmt->rowCount() > 0;
                    $tablesOk = $tablesOk && $exists;
                    echo '<div class="check ' . ($exists ? 'success' : 'error') . '">';
                    echo ($exists ? '✅' : '❌') . ' Table: ' . $table;
                    echo '</div>';
                }
                
                $allChecks[] = true;
                $allChecks[] = $tablesOk;
                
                // Show database info
                echo '<table>';
                echo '<tr><td>Host:</td><td>' . DB_HOST . '</td></tr>';
                echo '<tr><td>Database:</td><td>' . DB_NAME . '</td></tr>';
                echo '<tr><td>User:</td><td>' . DB_USER . '</td></tr>';
                echo '</table>';
                
            } catch (PDOException $e) {
                echo '<div class="check error">❌ Database connection failed: ' . htmlspecialchars($e->getMessage()) . '</div>';
                $allChecks[] = false;
            }
        } else {
            echo '<div class="check warning">⚠️ Cannot test database - configuration file missing</div>';
            $allChecks[] = false;
        }
        echo '</div>';

        // 5. Directory Permissions
        echo '<div class="section">';
        echo '<h2>5. Directory Permissions</h2>';
        $dirs = [
            'api/config' => 'Configuration directory',
            'api/uploads' => 'Uploads directory'
        ];
        $dirsOk = true;
        foreach ($dirs as $dir => $desc) {
            $path = __DIR__ . '/' . $dir;
            $exists = is_dir($path);
            $writable = $exists && is_writable($path);
            $dirsOk = $dirsOk && $writable;
            
            echo '<div class="check ' . ($writable ? 'success' : ($exists ? 'warning' : 'error')) . '">';
            if ($writable) {
                echo '✅ ' . $desc . ' exists and is writable';
            } elseif ($exists) {
                echo '⚠️ ' . $desc . ' exists but is not writable';
            } else {
                echo '❌ ' . $desc . ' does not exist';
            }
            echo '</div>';
        }
        $allChecks[] = $dirsOk;
        echo '</div>';

        // 6. File Structure
        echo '<div class="section">';
        echo '<h2>6. File Structure</h2>';
        $files = [
            'api/index.php' => 'API Entry Point',
            'api/config/database.php' => 'Database Class',
            'api/config/env.php' => 'Environment Config',
            'install.php' => 'Installation Wizard',
            'index.html' => 'Main Website'
        ];
        $filesOk = true;
        foreach ($files as $file => $desc) {
            $exists = file_exists(__DIR__ . '/' . $file);
            $filesOk = $filesOk && $exists;
            echo '<div class="check ' . ($exists ? 'success' : 'error') . '">';
            echo ($exists ? '✅' : '❌') . ' ' . $desc . ' (' . $file . ')';
            echo '</div>';
        }
        $allChecks[] = $filesOk;
        echo '</div>';

        // Summary
        echo '<div class="section">';
        echo '<h2>Summary</h2>';
        $allPassed = !in_array(false, $allChecks);
        echo '<div class="check ' . ($allPassed ? 'success' : 'error') . '" style="font-size: 1.2em; font-weight: bold; padding: 20px;">';
        if ($allPassed) {
            echo '✅ All checks passed! Your installation is ready.';
        } else {
            echo '❌ Some checks failed. Please review the issues above and fix them.';
        }
        echo '</div>';
        echo '</div>';

        // Quick Actions
        echo '<div class="section">';
        echo '<h2>Quick Actions</h2>';
        echo '<a href="install.php" class="btn">Run Installation Wizard</a>';
        echo '<a href="api/setup.php" class="btn">Check API Setup</a>';
        echo '<a href="index.html" class="btn">Visit Website</a>';
        echo '</div>';
        ?>
    </div>
</body>
</html>
