<?php
require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../utils/FileUpload.php';

class ProductController {
    private $db;
    private $fileUpload;
    
    public function __construct() {
        $this->db = Database::getInstance()->getConnection();
        $this->fileUpload = new FileUpload();
    }
    
    public function addPart() {
        try {
            // Validate required fields
            $requiredFields = ['partNumber', 'manufacturer', 'modality', 'product'];
            foreach ($requiredFields as $field) {
                if (empty($_POST[$field])) {
                    http_response_code(400);
                    echo json_encode(['message' => "Field '$field' is required"]);
                    return;
                }
            }
            
            // Handle image upload
            $imagePath = null;
            if (isset($_FILES['image']) && $_FILES['image']['error'] === UPLOAD_ERR_OK) {
                $imagePath = $this->fileUpload->uploadFile($_FILES['image']);
            } else {
                http_response_code(400);
                echo json_encode(['message' => 'Image is required']);
                return;
            }
            
            // Prepare data for database
            $data = [
                'part_number' => $_POST['partNumber'],
                'manufacturer' => $_POST['manufacturer'],
                'modality' => $_POST['modality'],
                'product' => $_POST['product'],
                'modal' => $_POST['modal'] ?? '',
                'image' => $imagePath
            ];
            
            // Insert into database
            $stmt = $this->db->prepare("
                INSERT INTO parts 
                (part_number, manufacturer, modality, product, modal, image) 
                VALUES (:part_number, :manufacturer, :modality, :product, :modal, :image)
            ");
            
            $stmt->execute($data);
            
            // Return success response with mapped field names
            $insertedId = $this->db->lastInsertId();
            http_response_code(201);
            echo json_encode([
                'success' => true,
                'message' => 'Part added successfully!',
                'part' => [
                    'id' => $insertedId,
                    'partNumber' => $data['part_number'],
                    'manufacturer' => $data['manufacturer'],
                    'modality' => $data['modality'],
                    'product' => $data['product'],
                    'modal' => $data['modal'],
                    'image' => $data['image']
                ]
            ]);
            
        } catch (Exception $e) {
            error_log('Add part error: ' . $e->getMessage());
            http_response_code(500);
            echo json_encode([
                'success' => false,
                'message' => 'Server error while adding part.',
                'error' => $e->getMessage()
            ]);
        }
    }
    
    public function getAllParts() {
        try {
            $stmt = $this->db->prepare("SELECT * FROM parts ORDER BY created_at DESC");
            $stmt->execute();
            $parts = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            // Map database field names to frontend expected field names
            $mappedParts = array_map(function($part) {
                return [
                    'id' => $part['id'],
                    'partNumber' => $part['part_number'],
                    'manufacturer' => $part['manufacturer'],
                    'modality' => $part['modality'],
                    'product' => $part['product'],
                    'modal' => $part['modal'],
                    'image' => $part['image'],
                    'created_at' => $part['created_at'],
                    'updated_at' => $part['updated_at']
                ];
            }, $parts);
            
            http_response_code(200);
            echo json_encode($mappedParts);
            
        } catch (Exception $e) {
            error_log('Get parts error: ' . $e->getMessage());
            http_response_code(500);
            echo json_encode([
                'success' => false,
                'message' => 'Failed to fetch parts',
                'error' => $e->getMessage()
            ]);
        }
    }
    
    public function getPartsByModality($modality) {
        try {
            $stmt = $this->db->prepare("SELECT * FROM parts WHERE modality = :modality ORDER BY created_at DESC");
            $stmt->execute(['modality' => $modality]);
            $parts = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            // Map database field names to frontend expected field names
            $mappedParts = array_map(function($part) {
                return [
                    'id' => $part['id'],
                    'partNumber' => $part['part_number'],
                    'manufacturer' => $part['manufacturer'],
                    'modality' => $part['modality'],
                    'product' => $part['product'],
                    'modal' => $part['modal'],
                    'image' => $part['image'],
                    'created_at' => $part['created_at'],
                    'updated_at' => $part['updated_at']
                ];
            }, $parts);
            
            http_response_code(200);
            echo json_encode($mappedParts);
            
        } catch (Exception $e) {
            error_log('Get parts by modality error: ' . $e->getMessage());
            http_response_code(500);
            echo json_encode(['error' => 'Failed to fetch parts']);
        }
    }
    
    public function updatePart($id) {
        try {
            // Check if part exists
            $stmt = $this->db->prepare("SELECT * FROM parts WHERE id = :id");
            $stmt->execute(['id' => $id]);
            $existingPart = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$existingPart) {
                http_response_code(404);
                echo json_encode(['success' => false, 'message' => 'Part not found']);
                return;
            }
            
            // Prepare update data
            $data = [
                'part_number' => $_POST['partNumber'] ?? $existingPart['part_number'],
                'manufacturer' => $_POST['manufacturer'] ?? $existingPart['manufacturer'],
                'modality' => $_POST['modality'] ?? $existingPart['modality'],
                'product' => $_POST['product'] ?? $existingPart['product'],
                'modal' => $_POST['modal'] ?? $existingPart['modal'],
                'id' => $id
            ];
            
            // Handle image upload if provided
            if (isset($_FILES['image']) && $_FILES['image']['error'] === UPLOAD_ERR_OK) {
                // Delete old image
                if ($existingPart['image']) {
                    $this->fileUpload->deleteFile($existingPart['image']);
                }
                // Upload new image
                $data['image'] = $this->fileUpload->uploadFile($_FILES['image']);
            } else {
                $data['image'] = $existingPart['image'];
            }
            
            // Update in database
            $stmt = $this->db->prepare("
                UPDATE parts 
                SET part_number = :part_number, manufacturer = :manufacturer, 
                    modality = :modality, product = :product, modal = :modal, image = :image 
                WHERE id = :id
            ");
            
            $stmt->execute($data);
            
            // Return success response with mapped field names
            http_response_code(200);
            echo json_encode([
                'success' => true,
                'message' => 'Part updated successfully',
                'part' => [
                    'id' => $data['id'],
                    'partNumber' => $data['part_number'],
                    'manufacturer' => $data['manufacturer'],
                    'modality' => $data['modality'],
                    'product' => $data['product'],
                    'modal' => $data['modal'],
                    'image' => $data['image']
                ]
            ]);
            
        } catch (Exception $e) {
            error_log('Update part error: ' . $e->getMessage());
            http_response_code(500);
            echo json_encode([
                'success' => false,
                'message' => 'Error updating part',
                'error' => $e->getMessage()
            ]);
        }
    }
    
    public function deletePart($id) {
        try {
            // Get part info first
            $stmt = $this->db->prepare("SELECT * FROM parts WHERE id = :id");
            $stmt->execute(['id' => $id]);
            $part = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$part) {
                http_response_code(404);
                echo json_encode(['success' => false, 'message' => 'Part not found']);
                return;
            }
            
            // Delete the part from database
            $stmt = $this->db->prepare("DELETE FROM parts WHERE id = :id");
            $stmt->execute(['id' => $id]);
            
            // Delete associated image file
            if ($part['image']) {
                $this->fileUpload->deleteFile($part['image']);
            }
            
            // Return success response
            http_response_code(200);
            echo json_encode([
                'success' => true,
                'message' => 'Part deleted successfully'
            ]);
            
        } catch (Exception $e) {
            error_log('Delete part error: ' . $e->getMessage());
            http_response_code(500);
            echo json_encode([
                'success' => false,
                'message' => 'Error deleting part',
                'error' => $e->getMessage()
            ]);
        }
    }
}
?>
