<?php
require_once __DIR__ . '/../config/env.php';

class FileUpload {
    private $uploadPath;
    private $uploadUrl;
    private $maxFileSize;
    private $allowedExtensions;
    
    public function __construct() {
        $this->uploadPath = UPLOAD_PATH;
        $this->uploadUrl = UPLOAD_URL;
        $this->maxFileSize = MAX_FILE_SIZE;
        $this->allowedExtensions = ALLOWED_EXTENSIONS;
        
        // Create upload directory if it doesn't exist
        if (!is_dir($this->uploadPath)) {
            mkdir($this->uploadPath, 0755, true);
        }
    }
    
    public function uploadFile($file) {
        try {
            // Check if file was uploaded without errors
            if ($file['error'] !== UPLOAD_ERR_OK) {
                throw new Exception('File upload error: ' . $this->getUploadErrorMessage($file['error']));
            }
            
            // Check file size
            if ($file['size'] > $this->maxFileSize) {
                throw new Exception('File size exceeds maximum allowed size');
            }
            
            // Get file extension
            $extension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
            
            // Check file extension
            if (!in_array($extension, $this->allowedExtensions)) {
                throw new Exception('File type not allowed');
            }
            
            // Generate unique filename
            $filename = time() . '-' . uniqid() . '.' . $extension;
            $filepath = $this->uploadPath . $filename;
            
            // Move uploaded file
            if (!move_uploaded_file($file['tmp_name'], $filepath)) {
                throw new Exception('Failed to move uploaded file');
            }
            
            // Return the URL path
            return $this->uploadUrl . $filename;
        } catch (Exception $e) {
            error_log('File upload error: ' . $e->getMessage());
            throw $e;
        }
    }
    
    public function deleteFile($filepath) {
        try {
            $filename = basename($filepath);
            $fullPath = $this->uploadPath . $filename;
            
            if (file_exists($fullPath)) {
                return unlink($fullPath);
            }
            
            return false;
        } catch (Exception $e) {
            error_log('File deletion error: ' . $e->getMessage());
            return false;
        }
    }
    
    private function getUploadErrorMessage($errorCode) {
        switch ($errorCode) {
            case UPLOAD_ERR_INI_SIZE:
                return 'The uploaded file exceeds the upload_max_filesize directive in php.ini';
            case UPLOAD_ERR_FORM_SIZE:
                return 'The uploaded file exceeds the MAX_FILE_SIZE directive that was specified in the HTML form';
            case UPLOAD_ERR_PARTIAL:
                return 'The uploaded file was only partially uploaded';
            case UPLOAD_ERR_NO_FILE:
                return 'No file was uploaded';
            case UPLOAD_ERR_NO_TMP_DIR:
                return 'Missing a temporary folder';
            case UPLOAD_ERR_CANT_WRITE:
                return 'Failed to write file to disk';
            case UPLOAD_ERR_EXTENSION:
                return 'A PHP extension stopped the file upload';
            default:
                return 'Unknown upload error';
        }
    }
}
?>
